// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package iap provides access to the Cloud Identity-Aware Proxy API.
//
// For product documentation, see: https://cloud.google.com/iap
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/iap/v1"
//	...
//	ctx := context.Background()
//	iapService, err := iap.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	iapService, err := iap.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	iapService, err := iap.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package iap // import "google.golang.org/api/iap/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "iap:v1"
const apiName = "iap"
const apiVersion = "v1"
const basePath = "https://iap.googleapis.com/"
const basePathTemplate = "https://iap.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://iap.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, configure, and delete your Google Cloud data and see the email
	// address for your Google Account.
	CloudPlatformScope = "https://www.googleapis.com/auth/cloud-platform"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/cloud-platform",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Projects = NewProjectsService(s)
	s.V1 = NewV1Service(s)
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Projects *ProjectsService

	V1 *V1Service
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewProjectsService(s *Service) *ProjectsService {
	rs := &ProjectsService{s: s}
	rs.Brands = NewProjectsBrandsService(s)
	rs.IapTunnel = NewProjectsIapTunnelService(s)
	return rs
}

type ProjectsService struct {
	s *Service

	Brands *ProjectsBrandsService

	IapTunnel *ProjectsIapTunnelService
}

func NewProjectsBrandsService(s *Service) *ProjectsBrandsService {
	rs := &ProjectsBrandsService{s: s}
	rs.IdentityAwareProxyClients = NewProjectsBrandsIdentityAwareProxyClientsService(s)
	return rs
}

type ProjectsBrandsService struct {
	s *Service

	IdentityAwareProxyClients *ProjectsBrandsIdentityAwareProxyClientsService
}

func NewProjectsBrandsIdentityAwareProxyClientsService(s *Service) *ProjectsBrandsIdentityAwareProxyClientsService {
	rs := &ProjectsBrandsIdentityAwareProxyClientsService{s: s}
	return rs
}

type ProjectsBrandsIdentityAwareProxyClientsService struct {
	s *Service
}

func NewProjectsIapTunnelService(s *Service) *ProjectsIapTunnelService {
	rs := &ProjectsIapTunnelService{s: s}
	rs.Locations = NewProjectsIapTunnelLocationsService(s)
	return rs
}

type ProjectsIapTunnelService struct {
	s *Service

	Locations *ProjectsIapTunnelLocationsService
}

func NewProjectsIapTunnelLocationsService(s *Service) *ProjectsIapTunnelLocationsService {
	rs := &ProjectsIapTunnelLocationsService{s: s}
	rs.DestGroups = NewProjectsIapTunnelLocationsDestGroupsService(s)
	return rs
}

type ProjectsIapTunnelLocationsService struct {
	s *Service

	DestGroups *ProjectsIapTunnelLocationsDestGroupsService
}

func NewProjectsIapTunnelLocationsDestGroupsService(s *Service) *ProjectsIapTunnelLocationsDestGroupsService {
	rs := &ProjectsIapTunnelLocationsDestGroupsService{s: s}
	return rs
}

type ProjectsIapTunnelLocationsDestGroupsService struct {
	s *Service
}

func NewV1Service(s *Service) *V1Service {
	rs := &V1Service{s: s}
	return rs
}

type V1Service struct {
	s *Service
}

// AccessDeniedPageSettings: Custom content configuration for access denied
// page. IAP allows customers to define a custom URI to use as the error page
// when access is denied to users. If IAP prevents access to this page, the
// default IAP error page will be displayed instead.
type AccessDeniedPageSettings struct {
	// AccessDeniedPageUri: The URI to be redirected to when access is denied.
	AccessDeniedPageUri string `json:"accessDeniedPageUri,omitempty"`
	// GenerateTroubleshootingUri: Whether to generate a troubleshooting URL on
	// access denied events to this application.
	GenerateTroubleshootingUri bool `json:"generateTroubleshootingUri,omitempty"`
	// RemediationTokenGenerationEnabled: Whether to generate remediation token on
	// access denied events to this application.
	RemediationTokenGenerationEnabled bool `json:"remediationTokenGenerationEnabled,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccessDeniedPageUri") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessDeniedPageUri") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessDeniedPageSettings) MarshalJSON() ([]byte, error) {
	type NoMethod AccessDeniedPageSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AccessSettings: Access related settings for IAP protected apps.
type AccessSettings struct {
	// AllowedDomainsSettings: Optional. Settings to configure and enable allowed
	// domains.
	AllowedDomainsSettings *AllowedDomainsSettings `json:"allowedDomainsSettings,omitempty"`
	// CorsSettings: Optional. Configuration to allow cross-origin requests via
	// IAP.
	CorsSettings *CorsSettings `json:"corsSettings,omitempty"`
	// GcipSettings: Optional. GCIP claims and endpoint configurations for 3p
	// identity providers.
	GcipSettings *GcipSettings `json:"gcipSettings,omitempty"`
	// IdentitySources: Optional. Identity sources that IAP can use to authenticate
	// the end user. Only one identity source can be configured.
	//
	// Possible values:
	//   "IDENTITY_SOURCE_UNSPECIFIED" - IdentitySource Unspecified. When selected,
	// IAP relies on which identity settings are fully configured to redirect the
	// traffic to. The precedence order is WorkforceIdentitySettings >
	// GcipSettings. If none is set, default to use Google identity.
	//   "WORKFORCE_IDENTITY_FEDERATION" - Use external identities set up on Google
	// Cloud Workforce Identity Federation.
	IdentitySources []string `json:"identitySources,omitempty"`
	// OauthSettings: Optional. Settings to configure IAP's OAuth behavior.
	OauthSettings *OAuthSettings `json:"oauthSettings,omitempty"`
	// PolicyDelegationSettings: Optional. Settings to allow google-internal teams
	// to use IAP for apps hosted in a tenant project.
	PolicyDelegationSettings *PolicyDelegationSettings `json:"policyDelegationSettings,omitempty"`
	// ReauthSettings: Optional. Settings to configure reauthentication policies in
	// IAP.
	ReauthSettings *ReauthSettings `json:"reauthSettings,omitempty"`
	// WorkforceIdentitySettings: Optional. Settings to configure the workforce
	// identity federation, including workforce pools and OAuth 2.0 settings.
	WorkforceIdentitySettings *WorkforceIdentitySettings `json:"workforceIdentitySettings,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AllowedDomainsSettings") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AllowedDomainsSettings") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessSettings) MarshalJSON() ([]byte, error) {
	type NoMethod AccessSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AllowedDomainsSettings: Configuration for IAP allowed domains. Lets you to
// restrict access to an app and allow access to only the domains that you
// list.
type AllowedDomainsSettings struct {
	// Domains: Optional. List of trusted domains.
	Domains []string `json:"domains,omitempty"`
	// Enable: Optional. Configuration for customers to opt in for the feature.
	Enable bool `json:"enable,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Domains") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Domains") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AllowedDomainsSettings) MarshalJSON() ([]byte, error) {
	type NoMethod AllowedDomainsSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ApplicationSettings: Wrapper over application specific settings for IAP.
type ApplicationSettings struct {
	// AccessDeniedPageSettings: Optional. Customization for Access Denied page.
	AccessDeniedPageSettings *AccessDeniedPageSettings `json:"accessDeniedPageSettings,omitempty"`
	// AttributePropagationSettings: Optional. Settings to configure attribute
	// propagation.
	AttributePropagationSettings *AttributePropagationSettings `json:"attributePropagationSettings,omitempty"`
	// CookieDomain: The Domain value to set for cookies generated by IAP. This
	// value is not validated by the API, but will be ignored at runtime if
	// invalid.
	CookieDomain string `json:"cookieDomain,omitempty"`
	// CsmSettings: Optional. Settings to configure IAP's behavior for a service
	// mesh.
	CsmSettings *CsmSettings `json:"csmSettings,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AccessDeniedPageSettings")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessDeniedPageSettings") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ApplicationSettings) MarshalJSON() ([]byte, error) {
	type NoMethod ApplicationSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AttributePropagationSettings: Configuration for propagating attributes to
// applications protected by IAP.
type AttributePropagationSettings struct {
	// Enable: Optional. Whether the provided attribute propagation settings should
	// be evaluated on user requests. If set to true, attributes returned from the
	// expression will be propagated in the set output credentials.
	Enable bool `json:"enable,omitempty"`
	// Expression: Optional. Raw string CEL expression. Must return a list of
	// attributes. A maximum of 45 attributes can be selected. Expressions can
	// select different attribute types from `attributes`:
	// `attributes.saml_attributes`, `attributes.iap_attributes`. The following
	// functions are supported: - filter `.filter(, )`: Returns a subset of ``
	// where `` is true for every item. - in ` in `: Returns true if `` contains
	// ``. - selectByName `.selectByName()`: Returns the attribute in `` with the
	// given `` name, otherwise returns empty. - emitAs `.emitAs()`: Sets the ``
	// name field to the given `` for propagation in selected output credentials. -
	// strict `.strict()`: Ignores the `x-goog-iap-attr-` prefix for the provided
	// `` when propagating with the `HEADER` output credential, such as request
	// headers. - append `.append()` OR `.append()`: Appends the provided `` or ``
	// to the end of ``. Example expression: `attributes.saml_attributes.filter(x,
	// x.name in
	// ['test']).append(attributes.iap_attributes.selectByName('exact').emitAs('cust
	// om').strict())`
	Expression string `json:"expression,omitempty"`
	// OutputCredentials: Optional. Which output credentials attributes selected by
	// the CEL expression should be propagated in. All attributes will be fully
	// duplicated in each selected output credential.
	//
	// Possible values:
	//   "OUTPUT_CREDENTIALS_UNSPECIFIED" - An output credential is required.
	//   "HEADER" - Propagate attributes in the headers with "x-goog-iap-attr-"
	// prefix.
	//   "JWT" - Propagate attributes in the JWT of the form: "additional_claims":
	// { "my_attribute": ["value1", "value2"] }`
	//   "RCTOKEN" - Propagate attributes in the RCToken of the form:
	// "additional_claims": { "my_attribute": ["value1", "value2"] }`
	OutputCredentials []string `json:"outputCredentials,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Enable") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Enable") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AttributePropagationSettings) MarshalJSON() ([]byte, error) {
	type NoMethod AttributePropagationSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Binding: Associates `members`, or principals, with a `role`.
type Binding struct {
	// Condition: The condition that is associated with this binding. If the
	// condition evaluates to `true`, then this binding applies to the current
	// request. If the condition evaluates to `false`, then this binding does not
	// apply to the current request. However, a different role binding might grant
	// the same role to one or more of the principals in this binding. To learn
	// which resources support conditions in their IAM policies, see the IAM
	// documentation
	// (https://cloud.google.com/iam/help/conditions/resource-policies).
	Condition *Expr `json:"condition,omitempty"`
	// Members: Specifies the principals requesting access for a Google Cloud
	// resource. `members` can have the following values: * `allUsers`: A special
	// identifier that represents anyone who is on the internet; with or without a
	// Google account. * `allAuthenticatedUsers`: A special identifier that
	// represents anyone who is authenticated with a Google account or a service
	// account. Does not include identities that come from external identity
	// providers (IdPs) through identity federation. * `user:{emailid}`: An email
	// address that represents a specific Google account. For example,
	// `alice@example.com` . * `serviceAccount:{emailid}`: An email address that
	// represents a Google service account. For example,
	// `my-other-app@appspot.gserviceaccount.com`. *
	// `serviceAccount:{projectid}.svc.id.goog[{namespace}/{kubernetes-sa}]`: An
	// identifier for a Kubernetes service account
	// (https://cloud.google.com/kubernetes-engine/docs/how-to/kubernetes-service-accounts).
	// For example, `my-project.svc.id.goog[my-namespace/my-kubernetes-sa]`. *
	// `group:{emailid}`: An email address that represents a Google group. For
	// example, `admins@example.com`. * `domain:{domain}`: The G Suite domain
	// (primary) that represents all the users of that domain. For example,
	// `google.com` or `example.com`. *
	// `principal://iam.googleapis.com/locations/global/workforcePools/{pool_id}/sub
	// ject/{subject_attribute_value}`: A single identity in a workforce identity
	// pool. *
	// `principalSet://iam.googleapis.com/locations/global/workforcePools/{pool_id}/
	// group/{group_id}`: All workforce identities in a group. *
	// `principalSet://iam.googleapis.com/locations/global/workforcePools/{pool_id}/
	// attribute.{attribute_name}/{attribute_value}`: All workforce identities with
	// a specific attribute value. *
	// `principalSet://iam.googleapis.com/locations/global/workforcePools/{pool_id}/
	// *`: All identities in a workforce identity pool. *
	// `principal://iam.googleapis.com/projects/{project_number}/locations/global/wo
	// rkloadIdentityPools/{pool_id}/subject/{subject_attribute_value}`: A single
	// identity in a workload identity pool. *
	// `principalSet://iam.googleapis.com/projects/{project_number}/locations/global
	// /workloadIdentityPools/{pool_id}/group/{group_id}`: A workload identity pool
	// group. *
	// `principalSet://iam.googleapis.com/projects/{project_number}/locations/global
	// /workloadIdentityPools/{pool_id}/attribute.{attribute_name}/{attribute_value}
	// `: All identities in a workload identity pool with a certain attribute. *
	// `principalSet://iam.googleapis.com/projects/{project_number}/locations/global
	// /workloadIdentityPools/{pool_id}/*`: All identities in a workload identity
	// pool. * `deleted:user:{emailid}?uid={uniqueid}`: An email address (plus
	// unique identifier) representing a user that has been recently deleted. For
	// example, `alice@example.com?uid=123456789012345678901`. If the user is
	// recovered, this value reverts to `user:{emailid}` and the recovered user
	// retains the role in the binding. *
	// `deleted:serviceAccount:{emailid}?uid={uniqueid}`: An email address (plus
	// unique identifier) representing a service account that has been recently
	// deleted. For example,
	// `my-other-app@appspot.gserviceaccount.com?uid=123456789012345678901`. If the
	// service account is undeleted, this value reverts to
	// `serviceAccount:{emailid}` and the undeleted service account retains the
	// role in the binding. * `deleted:group:{emailid}?uid={uniqueid}`: An email
	// address (plus unique identifier) representing a Google group that has been
	// recently deleted. For example,
	// `admins@example.com?uid=123456789012345678901`. If the group is recovered,
	// this value reverts to `group:{emailid}` and the recovered group retains the
	// role in the binding. *
	// `deleted:principal://iam.googleapis.com/locations/global/workforcePools/{pool
	// _id}/subject/{subject_attribute_value}`: Deleted single identity in a
	// workforce identity pool. For example,
	// `deleted:principal://iam.googleapis.com/locations/global/workforcePools/my-po
	// ol-id/subject/my-subject-attribute-value`.
	Members []string `json:"members,omitempty"`
	// Role: Role that is assigned to the list of `members`, or principals. For
	// example, `roles/viewer`, `roles/editor`, or `roles/owner`. For an overview
	// of the IAM roles and permissions, see the IAM documentation
	// (https://cloud.google.com/iam/docs/roles-overview). For a list of the
	// available pre-defined roles, see here
	// (https://cloud.google.com/iam/docs/understanding-roles).
	Role string `json:"role,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Condition") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Condition") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Binding) MarshalJSON() ([]byte, error) {
	type NoMethod Binding
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Brand: OAuth brand data. NOTE: Only contains a portion of the data that
// describes a brand.
type Brand struct {
	// ApplicationTitle: Application name displayed on OAuth consent screen.
	ApplicationTitle string `json:"applicationTitle,omitempty"`
	// Name: Output only. Identifier of the brand. NOTE: GCP project number
	// achieves the same brand identification purpose as only one brand per project
	// can be created.
	Name string `json:"name,omitempty"`
	// OrgInternalOnly: Output only. Whether the brand is only intended for usage
	// inside the G Suite organization only.
	OrgInternalOnly bool `json:"orgInternalOnly,omitempty"`
	// SupportEmail: Support email displayed on the OAuth consent screen.
	SupportEmail string `json:"supportEmail,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ApplicationTitle") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApplicationTitle") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Brand) MarshalJSON() ([]byte, error) {
	type NoMethod Brand
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CorsSettings: Allows customers to configure HTTP request paths that'll allow
// HTTP `OPTIONS` call to bypass authentication and authorization.
type CorsSettings struct {
	// AllowHttpOptions: Configuration to allow HTTP `OPTIONS` calls to skip
	// authentication and authorization. If undefined, IAP will not apply any
	// special logic to `OPTIONS` requests.
	AllowHttpOptions bool `json:"allowHttpOptions,omitempty"`
	// ForceSendFields is a list of field names (e.g. "AllowHttpOptions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AllowHttpOptions") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CorsSettings) MarshalJSON() ([]byte, error) {
	type NoMethod CorsSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// CsmSettings: Configuration for RCToken generated for service mesh workloads
// protected by IAP. RCToken are IAP generated JWTs that can be verified at the
// application. The RCToken is primarily used for service mesh deployments, and
// can be scoped to a single mesh by configuring the audience field
// accordingly.
type CsmSettings struct {
	// RctokenAud: Audience claim set in the generated RCToken. This value is not
	// validated by IAP.
	RctokenAud string `json:"rctokenAud,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RctokenAud") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RctokenAud") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s CsmSettings) MarshalJSON() ([]byte, error) {
	type NoMethod CsmSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// Expr: Represents a textual expression in the Common Expression Language
// (CEL) syntax. CEL is a C-like expression language. The syntax and semantics
// of CEL are documented at https://github.com/google/cel-spec. Example
// (Comparison): title: "Summary size limit" description: "Determines if a
// summary is less than 100 chars" expression: "document.summary.size() < 100"
// Example (Equality): title: "Requestor is owner" description: "Determines if
// requestor is the document owner" expression: "document.owner ==
// request.auth.claims.email" Example (Logic): title: "Public documents"
// description: "Determine whether the document should be publicly visible"
// expression: "document.type != 'private' && document.type != 'internal'"
// Example (Data Manipulation): title: "Notification string" description:
// "Create a notification string with a timestamp." expression: "'New message
// received at ' + string(document.create_time)" The exact variables and
// functions that may be referenced within an expression are determined by the
// service that evaluates it. See the service documentation for additional
// information.
type Expr struct {
	// Description: Optional. Description of the expression. This is a longer text
	// which describes the expression, e.g. when hovered over it in a UI.
	Description string `json:"description,omitempty"`
	// Expression: Textual representation of an expression in Common Expression
	// Language syntax.
	Expression string `json:"expression,omitempty"`
	// Location: Optional. String indicating the location of the expression for
	// error reporting, e.g. a file name and a position in the file.
	Location string `json:"location,omitempty"`
	// Title: Optional. Title for the expression, i.e. a short string describing
	// its purpose. This can be used e.g. in UIs which allow to enter the
	// expression.
	Title string `json:"title,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Description") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Description") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Expr) MarshalJSON() ([]byte, error) {
	type NoMethod Expr
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GcipSettings: Allows customers to configure tenant IDs for a Cloud Identity
// Platform (GCIP) instance for each application.
type GcipSettings struct {
	// LoginPageUri: Login page URI associated with the GCIP tenants. Typically,
	// all resources within the same project share the same login page, though it
	// could be overridden at the sub resource level.
	LoginPageUri string `json:"loginPageUri,omitempty"`
	// TenantIds: Optional. GCIP tenant IDs that are linked to the IAP resource.
	// `tenant_ids` could be a string beginning with a number character to indicate
	// authenticating with GCIP tenant flow, or in the format of `_` to indicate
	// authenticating with GCIP agent flow. If agent flow is used, `tenant_ids`
	// should only contain one single element, while for tenant flow, `tenant_ids`
	// can contain multiple elements.
	TenantIds []string `json:"tenantIds,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LoginPageUri") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LoginPageUri") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GcipSettings) MarshalJSON() ([]byte, error) {
	type NoMethod GcipSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GetIamPolicyRequest: Request message for `GetIamPolicy` method.
type GetIamPolicyRequest struct {
	// Options: OPTIONAL: A `GetPolicyOptions` object for specifying options to
	// `GetIamPolicy`.
	Options *GetPolicyOptions `json:"options,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Options") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Options") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GetIamPolicyRequest) MarshalJSON() ([]byte, error) {
	type NoMethod GetIamPolicyRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// GetPolicyOptions: Encapsulates settings provided to GetIamPolicy.
type GetPolicyOptions struct {
	// RequestedPolicyVersion: Optional. The maximum policy version that will be
	// used to format the policy. Valid values are 0, 1, and 3. Requests specifying
	// an invalid value will be rejected. Requests for policies with any
	// conditional role bindings must specify version 3. Policies with no
	// conditional role bindings may specify any valid value or leave the field
	// unset. The policy in the response might use the policy version that you
	// specified, or it might use a lower policy version. For example, if you
	// specify version 3, but the policy has no conditional role bindings, the
	// response uses version 1. To learn which resources support conditions in
	// their IAM policies, see the IAM documentation
	// (https://cloud.google.com/iam/help/conditions/resource-policies).
	RequestedPolicyVersion int64 `json:"requestedPolicyVersion,omitempty"`
	// ForceSendFields is a list of field names (e.g. "RequestedPolicyVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "RequestedPolicyVersion") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s GetPolicyOptions) MarshalJSON() ([]byte, error) {
	type NoMethod GetPolicyOptions
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IapSettings: The IAP configurable settings.
type IapSettings struct {
	// AccessSettings: Optional. Top level wrapper for all access related setting
	// in IAP
	AccessSettings *AccessSettings `json:"accessSettings,omitempty"`
	// ApplicationSettings: Optional. Top level wrapper for all application related
	// settings in IAP
	ApplicationSettings *ApplicationSettings `json:"applicationSettings,omitempty"`
	// Name: Required. The resource name of the IAP protected resource.
	Name string `json:"name,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccessSettings") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccessSettings") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IapSettings) MarshalJSON() ([]byte, error) {
	type NoMethod IapSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// IdentityAwareProxyClient: Contains the data that describes an Identity Aware
// Proxy owned client.
type IdentityAwareProxyClient struct {
	// DisplayName: Human-friendly name given to the OAuth client.
	DisplayName string `json:"displayName,omitempty"`
	// Name: Output only. Unique identifier of the OAuth client.
	Name string `json:"name,omitempty"`
	// Secret: Output only. Client secret of the OAuth client.
	Secret string `json:"secret,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "DisplayName") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DisplayName") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s IdentityAwareProxyClient) MarshalJSON() ([]byte, error) {
	type NoMethod IdentityAwareProxyClient
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListBrandsResponse: Response message for ListBrands.
type ListBrandsResponse struct {
	// Brands: Brands existing in the project.
	Brands []*Brand `json:"brands,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Brands") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Brands") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListBrandsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListBrandsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListIdentityAwareProxyClientsResponse: Response message for
// ListIdentityAwareProxyClients.
type ListIdentityAwareProxyClientsResponse struct {
	// IdentityAwareProxyClients: Clients existing in the brand.
	IdentityAwareProxyClients []*IdentityAwareProxyClient `json:"identityAwareProxyClients,omitempty"`
	// NextPageToken: A token, which can be send as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "IdentityAwareProxyClients")
	// to unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IdentityAwareProxyClients") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListIdentityAwareProxyClientsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListIdentityAwareProxyClientsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ListTunnelDestGroupsResponse: The response from ListTunnelDestGroups.
type ListTunnelDestGroupsResponse struct {
	// NextPageToken: A token that you can send as `page_token` to retrieve the
	// next page. If this field is omitted, there are no subsequent pages.
	NextPageToken string `json:"nextPageToken,omitempty"`
	// TunnelDestGroups: TunnelDestGroup existing in the project.
	TunnelDestGroups []*TunnelDestGroup `json:"tunnelDestGroups,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "NextPageToken") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "NextPageToken") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListTunnelDestGroupsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListTunnelDestGroupsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// NextStateOfTags: Used for calculating the next state of tags on the resource
// being passed for the CheckCustomConstraints RPC call. The detail evaluation
// of each field is described in go/op-create-update-time-tags and
// go/tags-in-orgpolicy-requests.
type NextStateOfTags struct {
	TagsFullState                 *TagsFullState                 `json:"tagsFullState,omitempty"`
	TagsFullStateForChildResource *TagsFullStateForChildResource `json:"tagsFullStateForChildResource,omitempty"`
	TagsPartialState              *TagsPartialState              `json:"tagsPartialState,omitempty"`
	// ForceSendFields is a list of field names (e.g. "TagsFullState") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "TagsFullState") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s NextStateOfTags) MarshalJSON() ([]byte, error) {
	type NoMethod NextStateOfTags
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// OAuth2: The OAuth 2.0 Settings
type OAuth2 struct {
	// ClientId: The OAuth 2.0 client ID registered in the workforce identity
	// federation OAuth 2.0 Server.
	ClientId string `json:"clientId,omitempty"`
	// ClientSecret: Input only. The OAuth 2.0 client secret created while
	// registering the client ID.
	ClientSecret string `json:"clientSecret,omitempty"`
	// ClientSecretSha256: Output only. SHA256 hash value for the client secret.
	// This field is returned by IAP when the settings are retrieved.
	ClientSecretSha256 string `json:"clientSecretSha256,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ClientId") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ClientId") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s OAuth2) MarshalJSON() ([]byte, error) {
	type NoMethod OAuth2
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// OAuthSettings: Configuration for OAuth login&consent flow behavior as well
// as for OAuth Credentials.
type OAuthSettings struct {
	// LoginHint: Domain hint to send as hd=? parameter in OAuth request flow.
	// Enables redirect to primary IDP by skipping Google's login screen.
	// https://developers.google.com/identity/protocols/OpenIDConnect#hd-param
	// Note: IAP does not verify that the id token's hd claim matches this value
	// since access behavior is managed by IAM policies.
	LoginHint string `json:"loginHint,omitempty"`
	// ProgrammaticClients: Optional. List of client ids allowed to use IAP
	// programmatically.
	ProgrammaticClients []string `json:"programmaticClients,omitempty"`
	// ForceSendFields is a list of field names (e.g. "LoginHint") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "LoginHint") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s OAuthSettings) MarshalJSON() ([]byte, error) {
	type NoMethod OAuthSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Policy: An Identity and Access Management (IAM) policy, which specifies
// access controls for Google Cloud resources. A `Policy` is a collection of
// `bindings`. A `binding` binds one or more `members`, or principals, to a
// single `role`. Principals can be user accounts, service accounts, Google
// groups, and domains (such as G Suite). A `role` is a named list of
// permissions; each `role` can be an IAM predefined role or a user-created
// custom role. For some types of Google Cloud resources, a `binding` can also
// specify a `condition`, which is a logical expression that allows access to a
// resource only if the expression evaluates to `true`. A condition can add
// constraints based on attributes of the request, the resource, or both. To
// learn which resources support conditions in their IAM policies, see the IAM
// documentation
// (https://cloud.google.com/iam/help/conditions/resource-policies). **JSON
// example:** ``` { "bindings": [ { "role":
// "roles/resourcemanager.organizationAdmin", "members": [
// "user:mike@example.com", "group:admins@example.com", "domain:google.com",
// "serviceAccount:my-project-id@appspot.gserviceaccount.com" ] }, { "role":
// "roles/resourcemanager.organizationViewer", "members": [
// "user:eve@example.com" ], "condition": { "title": "expirable access",
// "description": "Does not grant access after Sep 2020", "expression":
// "request.time < timestamp('2020-10-01T00:00:00.000Z')", } } ], "etag":
// "BwWWja0YfJA=", "version": 3 } ``` **YAML example:** ``` bindings: -
// members: - user:mike@example.com - group:admins@example.com -
// domain:google.com - serviceAccount:my-project-id@appspot.gserviceaccount.com
// role: roles/resourcemanager.organizationAdmin - members: -
// user:eve@example.com role: roles/resourcemanager.organizationViewer
// condition: title: expirable access description: Does not grant access after
// Sep 2020 expression: request.time < timestamp('2020-10-01T00:00:00.000Z')
// etag: BwWWja0YfJA= version: 3 ``` For a description of IAM and its features,
// see the IAM documentation (https://cloud.google.com/iam/docs/).
type Policy struct {
	// Bindings: Associates a list of `members`, or principals, with a `role`.
	// Optionally, may specify a `condition` that determines how and when the
	// `bindings` are applied. Each of the `bindings` must contain at least one
	// principal. The `bindings` in a `Policy` can refer to up to 1,500 principals;
	// up to 250 of these principals can be Google groups. Each occurrence of a
	// principal counts towards these limits. For example, if the `bindings` grant
	// 50 different roles to `user:alice@example.com`, and not to any other
	// principal, then you can add another 1,450 principals to the `bindings` in
	// the `Policy`.
	Bindings []*Binding `json:"bindings,omitempty"`
	// Etag: `etag` is used for optimistic concurrency control as a way to help
	// prevent simultaneous updates of a policy from overwriting each other. It is
	// strongly suggested that systems make use of the `etag` in the
	// read-modify-write cycle to perform policy updates in order to avoid race
	// conditions: An `etag` is returned in the response to `getIamPolicy`, and
	// systems are expected to put that etag in the request to `setIamPolicy` to
	// ensure that their change will be applied to the same version of the policy.
	// **Important:** If you use IAM Conditions, you must include the `etag` field
	// whenever you call `setIamPolicy`. If you omit this field, then IAM allows
	// you to overwrite a version `3` policy with a version `1` policy, and all of
	// the conditions in the version `3` policy are lost.
	Etag string `json:"etag,omitempty"`
	// Version: Specifies the format of the policy. Valid values are `0`, `1`, and
	// `3`. Requests that specify an invalid value are rejected. Any operation that
	// affects conditional role bindings must specify version `3`. This requirement
	// applies to the following operations: * Getting a policy that includes a
	// conditional role binding * Adding a conditional role binding to a policy *
	// Changing a conditional role binding in a policy * Removing any role binding,
	// with or without a condition, from a policy that includes conditions
	// **Important:** If you use IAM Conditions, you must include the `etag` field
	// whenever you call `setIamPolicy`. If you omit this field, then IAM allows
	// you to overwrite a version `3` policy with a version `1` policy, and all of
	// the conditions in the version `3` policy are lost. If a policy does not
	// include any conditions, operations on that policy may specify any valid
	// version or leave the field unset. To learn which resources support
	// conditions in their IAM policies, see the IAM documentation
	// (https://cloud.google.com/iam/help/conditions/resource-policies).
	Version int64 `json:"version,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Bindings") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Bindings") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Policy) MarshalJSON() ([]byte, error) {
	type NoMethod Policy
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PolicyDelegationSettings: PolicyDelegationConfig allows google-internal
// teams to use IAP for apps hosted in a tenant project. Using these settings,
// the app can delegate permission check to happen against the linked customer
// project. This is only ever supposed to be used by google internal teams,
// hence the restriction on the proto.
type PolicyDelegationSettings struct {
	// IamPermission: Permission to check in IAM.
	IamPermission string `json:"iamPermission,omitempty"`
	// IamServiceName: The DNS name of the service (e.g.
	// "resourcemanager.googleapis.com"). This should be the domain name part of
	// the full resource names (see https://aip.dev/122#full-resource-names), which
	// is usually the same as IamServiceSpec.service of the service where the
	// resource type is defined.
	IamServiceName string `json:"iamServiceName,omitempty"`
	// PolicyName: Policy name to be checked
	PolicyName *PolicyName `json:"policyName,omitempty"`
	// Resource: IAM resource to check permission on
	Resource *Resource `json:"resource,omitempty"`
	// ForceSendFields is a list of field names (e.g. "IamPermission") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "IamPermission") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PolicyDelegationSettings) MarshalJSON() ([]byte, error) {
	type NoMethod PolicyDelegationSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// PolicyName: An internal name for an IAM policy, based on the resource to
// which the policy applies. Not to be confused with a resource's external full
// resource name. For more information on this distinction, see
// go/iam-full-resource-names.
type PolicyName struct {
	// Id: Identifies an instance of the type. ID format varies by type. The ID
	// format is defined in the IAM .service file that defines the type, either in
	// path_mapping or in a comment.
	Id string `json:"id,omitempty"`
	// Region: For Cloud IAM: The location of the Policy. Must be empty or "global"
	// for Policies owned by global IAM. Must name a region from
	// prodspec/cloud-iam-cloudspec for Regional IAM Policies, see
	// go/iam-faq#where-is-iam-currently-deployed. For Local IAM: This field should
	// be set to "local".
	Region string `json:"region,omitempty"`
	// Type: Resource type. Types are defined in IAM's .service files. Valid values
	// for type might be 'storage_buckets', 'compute_instances',
	// 'resourcemanager_customers', 'billing_accounts', etc.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Id") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Id") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s PolicyName) MarshalJSON() ([]byte, error) {
	type NoMethod PolicyName
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ReauthSettings: Configuration for IAP reauthentication policies.
type ReauthSettings struct {
	// MaxAge: Optional. Reauth session lifetime, how long before a user has to
	// reauthenticate again.
	MaxAge string `json:"maxAge,omitempty"`
	// Method: Optional. Reauth method requested.
	//
	// Possible values:
	//   "METHOD_UNSPECIFIED" - Reauthentication disabled.
	//   "LOGIN" - Prompts the user to log in again.
	//   "PASSWORD"
	//   "SECURE_KEY" - User must use their secure key 2nd factor device.
	//   "ENROLLED_SECOND_FACTORS" - User can use any enabled 2nd factor.
	Method string `json:"method,omitempty"`
	// PolicyType: Optional. How IAP determines the effective policy in cases of
	// hierarchical policies. Policies are merged from higher in the hierarchy to
	// lower in the hierarchy.
	//
	// Possible values:
	//   "POLICY_TYPE_UNSPECIFIED" - Default value. This value is unused.
	//   "MINIMUM" - This policy acts as a minimum to other policies, lower in the
	// hierarchy. Effective policy may only be the same or stricter.
	//   "DEFAULT" - This policy acts as a default if no other reauth policy is
	// set.
	PolicyType string `json:"policyType,omitempty"`
	// ForceSendFields is a list of field names (e.g. "MaxAge") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "MaxAge") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ReauthSettings) MarshalJSON() ([]byte, error) {
	type NoMethod ReauthSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ResetIdentityAwareProxyClientSecretRequest: The request sent to
// ResetIdentityAwareProxyClientSecret.
type ResetIdentityAwareProxyClientSecretRequest struct {
}

type Resource struct {
	// ExpectedNextState: The proto or JSON formatted expected next state of the
	// resource, wrapped in a google.protobuf.Any proto, against which the policy
	// rules are evaluated. Services not integrated with custom org policy can omit
	// this field. Services integrated with custom org policy must populate this
	// field for all requests where the API call changes the state of the resource.
	// Custom org policy backend uses these attributes to enforce custom org
	// policies. For create operations, GCP service is expected to pass resource
	// from customer request as is. For update/patch operations, GCP service is
	// expected to compute the next state with the patch provided by the user. See
	// go/federated-custom-org-policy-integration-guide for additional details.
	ExpectedNextState googleapi.RawMessage `json:"expectedNextState,omitempty"`
	// Labels: The service defined labels of the resource on which the conditions
	// will be evaluated. The semantics - including the key names - are vague to
	// IAM. If the effective condition has a reference to a `resource.labels[foo]`
	// construct, IAM consults with this map to retrieve the values associated with
	// `foo` key for Conditions evaluation. If the provided key is not found in the
	// labels map, the condition would evaluate to false. This field is in limited
	// use. If your intended use case is not expected to express resource.labels
	// attribute in IAM Conditions, leave this field empty. Before planning on
	// using this attribute please: * Read go/iam-conditions-labels-comm and ensure
	// your service can meet the data availability and management requirements. *
	// Talk to iam-conditions-eng@ about your use case.
	Labels map[string]string `json:"labels,omitempty"`
	// Locations: The locations of the resource. This field is used to determine
	// whether the request is compliant with Trust Boundaries. Usage: - If unset or
	// empty, the location of authorization is used as the target location. - For
	// global resources: use a single value of "global". - For
	// regional/multi-regional resources: use name of the GCP region(s) where the
	// resource exists (e.g., ["us-east1", "us-west1"]). For multi-regional
	// resources specify the name of each GCP region in the resource's
	// multi-region. NOTE: Only GCP cloud region names are supported -
	// go/cloud-region-names.
	Locations []string `json:"locations,omitempty"`
	// Name: The **relative** name of the resource, which is the URI path of the
	// resource without the leading "/". See
	// https://cloud.google.com/iam/docs/conditions-resource-attributes#resource-name
	// for examples used by other GCP Services. This field is **required** for
	// services integrated with resource-attribute-based IAM conditions and/or
	// CustomOrgPolicy. This field requires special handling for parents-only
	// permissions such as `create` and `list`. See the document linked below for
	// further details. See go/iam-conditions-sig-g3#populate-resource-attributes
	// for specific details on populating this field.
	Name string `json:"name,omitempty"`
	// NextStateOfTags: Used for calculating the next state of tags on the resource
	// being passed for Custom Org Policy enforcement. NOTE: Only one of the tags
	// representations (i.e. numeric or namespaced) should be populated. The input
	// tags will be converted to the same representation before the calculation.
	// This behavior intentionally may differ from other tags related fields in
	// CheckPolicy request, which may require both formats to be passed in.
	// IMPORTANT: If tags are unchanged, this field should not be set.
	NextStateOfTags *NextStateOfTags `json:"nextStateOfTags,omitempty"`
	// Service: The name of the service this resource belongs to. It is configured
	// using the official_service_name of the Service as defined in service
	// configurations under //configs/cloud/resourcetypes. For example, the
	// official_service_name of cloud resource manager service is set as
	// 'cloudresourcemanager.googleapis.com' according to
	// //configs/cloud/resourcetypes/google/cloud/resourcemanager/prod.yaml This
	// field is **required** for services integrated with resource-attribute-based
	// IAM conditions and/or CustomOrgPolicy. This field requires special handling
	// for parents-only permissions such as `create` and `list`. See the document
	// linked below for further details. See
	// go/iam-conditions-sig-g3#populate-resource-attributes for specific details
	// on populating this field.
	Service string `json:"service,omitempty"`
	// Type: The public resource type name of the resource. It is configured using
	// the official_name of the ResourceType as defined in service configurations
	// under //configs/cloud/resourcetypes. For example, the official_name for GCP
	// projects is set as 'cloudresourcemanager.googleapis.com/Project' according
	// to //configs/cloud/resourcetypes/google/cloud/resourcemanager/prod.yaml This
	// field is **required** for services integrated with resource-attribute-based
	// IAM conditions and/or CustomOrgPolicy. This field requires special handling
	// for parents-only permissions such as `create` and `list`. See the document
	// linked below for further details. See
	// go/iam-conditions-sig-g3#populate-resource-attributes for specific details
	// on populating this field.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ExpectedNextState") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ExpectedNextState") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s Resource) MarshalJSON() ([]byte, error) {
	type NoMethod Resource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SetIamPolicyRequest: Request message for `SetIamPolicy` method.
type SetIamPolicyRequest struct {
	// Policy: REQUIRED: The complete policy to be applied to the `resource`. The
	// size of the policy is limited to a few 10s of KB. An empty policy is a valid
	// policy but certain Google Cloud services (such as Projects) might reject
	// them.
	Policy *Policy `json:"policy,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Policy") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Policy") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SetIamPolicyRequest) MarshalJSON() ([]byte, error) {
	type NoMethod SetIamPolicyRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type TagsFullState struct {
	// Tags: If TagsFullState is initialized, the values in this field fully
	// represent all the tags in the next state (the current tag values are not
	// used). If tags.size() == 0, the next state of tags would be no tags for
	// evaluation purposes. Only one type of tags reference (numeric or namespace)
	// is required to be passed.
	Tags map[string]string `json:"tags,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Tags") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Tags") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TagsFullState) MarshalJSON() ([]byte, error) {
	type NoMethod TagsFullState
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type TagsFullStateForChildResource struct {
	// Tags: If TagsFullStateForChildResource is initialized, the values in this
	// field represent all the tags in the next state for the child resource. Only
	// one type of tags reference (numeric or namespace) is required to be passed.
	// IMPORTANT: This field should only be used when the target resource IAM
	// policy name is UNKNOWN and the resource's parent IAM policy name is being
	// passed in the request.
	Tags map[string]string `json:"tags,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Tags") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Tags") to include in API requests
	// with the JSON null value. By default, fields with empty values are omitted
	// from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TagsFullStateForChildResource) MarshalJSON() ([]byte, error) {
	type NoMethod TagsFullStateForChildResource
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type TagsPartialState struct {
	// TagKeysToRemove: Keys of the tags that should be removed for evaluation
	// purposes. IMPORTANT: Currently only numeric references are supported. Once
	// support for namespace references is added, both the tag references (numeric
	// and namespace) will be removed.
	TagKeysToRemove []string `json:"tagKeysToRemove,omitempty"`
	// TagsToUpsert: Tags that’ll be updated or added to the current state of
	// tags for evaluation purposes. If a key exists in both "tags_to_upsert" and
	// "tag_keys_to_remove", the one in "tag_keys_to_remove" is ignored. Only one
	// type of tags reference (numeric or namespace) is required to be passed.
	TagsToUpsert map[string]string `json:"tagsToUpsert,omitempty"`
	// ForceSendFields is a list of field names (e.g. "TagKeysToRemove") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "TagKeysToRemove") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TagsPartialState) MarshalJSON() ([]byte, error) {
	type NoMethod TagsPartialState
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TestIamPermissionsRequest: Request message for `TestIamPermissions` method.
type TestIamPermissionsRequest struct {
	// Permissions: The set of permissions to check for the `resource`. Permissions
	// with wildcards (such as `*` or `storage.*`) are not allowed. For more
	// information see IAM Overview
	// (https://cloud.google.com/iam/docs/overview#permissions).
	Permissions []string `json:"permissions,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Permissions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Permissions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TestIamPermissionsRequest) MarshalJSON() ([]byte, error) {
	type NoMethod TestIamPermissionsRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TestIamPermissionsResponse: Response message for `TestIamPermissions`
// method.
type TestIamPermissionsResponse struct {
	// Permissions: A subset of `TestPermissionsRequest.permissions` that the
	// caller is allowed.
	Permissions []string `json:"permissions,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Permissions") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Permissions") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TestIamPermissionsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod TestIamPermissionsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// TunnelDestGroup: A TunnelDestGroup.
type TunnelDestGroup struct {
	// Cidrs: Optional. Unordered list. List of CIDRs that this group applies to.
	Cidrs []string `json:"cidrs,omitempty"`
	// Fqdns: Optional. Unordered list. List of FQDNs that this group applies to.
	Fqdns []string `json:"fqdns,omitempty"`
	// Name: Identifier. Identifier for the TunnelDestGroup. Must be unique within
	// the project and contain only lower case letters (a-z) and dashes (-).
	Name string `json:"name,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Cidrs") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Cidrs") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s TunnelDestGroup) MarshalJSON() ([]byte, error) {
	type NoMethod TunnelDestGroup
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ValidateIapAttributeExpressionResponse: IAP Expression Linter endpoint
// returns empty response body.
type ValidateIapAttributeExpressionResponse struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// WorkforceIdentitySettings: WorkforceIdentitySettings allows customers to
// configure workforce pools and OAuth 2.0 settings to gate their applications
// using a third-party IdP with access control.
type WorkforceIdentitySettings struct {
	// Oauth2: OAuth 2.0 settings for IAP to perform OIDC flow with workforce
	// identity federation services.
	Oauth2 *OAuth2 `json:"oauth2,omitempty"`
	// WorkforcePools: The workforce pool resources. Only one workforce pool is
	// accepted.
	WorkforcePools []string `json:"workforcePools,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Oauth2") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Oauth2") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s WorkforceIdentitySettings) MarshalJSON() ([]byte, error) {
	type NoMethod WorkforceIdentitySettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type ProjectsBrandsCreateCall struct {
	s          *Service
	parent     string
	brand      *Brand
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Create: Constructs a new OAuth brand for the project if one does not exist.
// The created brand is "internal only", meaning that OAuth clients created
// under it only accept requests from users who belong to the same Google
// Workspace organization as the project. The brand is created in an
// un-reviewed status. NOTE: The "internal only" status can be manually changed
// in the Google Cloud Console. Requires that a brand does not already exist
// for the project, and that the specified support email is owned by the
// caller.
//
//   - parent: GCP Project number/id under which the brand is to be created. In
//     the following format: projects/{project_number/id}.
func (r *ProjectsBrandsService) Create(parent string, brand *Brand) *ProjectsBrandsCreateCall {
	c := &ProjectsBrandsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.brand = brand
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsBrandsCreateCall) Fields(s ...googleapi.Field) *ProjectsBrandsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsBrandsCreateCall) Context(ctx context.Context) *ProjectsBrandsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsBrandsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsBrandsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.brand)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/brands")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.brands.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.brands.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *Brand.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsBrandsCreateCall) Do(opts ...googleapi.CallOption) (*Brand, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Brand{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.brands.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsBrandsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves the OAuth brand of the project.
//
//   - name: Name of the brand to be fetched. In the following format:
//     projects/{project_number/id}/brands/{brand}.
func (r *ProjectsBrandsService) Get(name string) *ProjectsBrandsGetCall {
	c := &ProjectsBrandsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsBrandsGetCall) Fields(s ...googleapi.Field) *ProjectsBrandsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsBrandsGetCall) IfNoneMatch(entityTag string) *ProjectsBrandsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsBrandsGetCall) Context(ctx context.Context) *ProjectsBrandsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsBrandsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsBrandsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.brands.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.brands.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *Brand.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsBrandsGetCall) Do(opts ...googleapi.CallOption) (*Brand, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Brand{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.brands.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsBrandsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the existing brands for the project.
//
//   - parent: GCP Project number/id. In the following format:
//     projects/{project_number/id}.
func (r *ProjectsBrandsService) List(parent string) *ProjectsBrandsListCall {
	c := &ProjectsBrandsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsBrandsListCall) Fields(s ...googleapi.Field) *ProjectsBrandsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsBrandsListCall) IfNoneMatch(entityTag string) *ProjectsBrandsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsBrandsListCall) Context(ctx context.Context) *ProjectsBrandsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsBrandsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsBrandsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/brands")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.brands.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.brands.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListBrandsResponse.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsBrandsListCall) Do(opts ...googleapi.CallOption) (*ListBrandsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListBrandsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.brands.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsBrandsIdentityAwareProxyClientsCreateCall struct {
	s                        *Service
	parent                   string
	identityawareproxyclient *IdentityAwareProxyClient
	urlParams_               gensupport.URLParams
	ctx_                     context.Context
	header_                  http.Header
}

// Create: Creates an Identity Aware Proxy (IAP) OAuth client. The client is
// owned by IAP. Requires that the brand for the project exists and that it is
// set for internal-only use.
//
//   - parent: Path to create the client in. In the following format:
//     projects/{project_number/id}/brands/{brand}. The project must belong to a
//     G Suite account.
func (r *ProjectsBrandsIdentityAwareProxyClientsService) Create(parent string, identityawareproxyclient *IdentityAwareProxyClient) *ProjectsBrandsIdentityAwareProxyClientsCreateCall {
	c := &ProjectsBrandsIdentityAwareProxyClientsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.identityawareproxyclient = identityawareproxyclient
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsBrandsIdentityAwareProxyClientsCreateCall) Fields(s ...googleapi.Field) *ProjectsBrandsIdentityAwareProxyClientsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsBrandsIdentityAwareProxyClientsCreateCall) Context(ctx context.Context) *ProjectsBrandsIdentityAwareProxyClientsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsBrandsIdentityAwareProxyClientsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsBrandsIdentityAwareProxyClientsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.identityawareproxyclient)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/identityAwareProxyClients")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.brands.identityAwareProxyClients.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *IdentityAwareProxyClient.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsBrandsIdentityAwareProxyClientsCreateCall) Do(opts ...googleapi.CallOption) (*IdentityAwareProxyClient, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &IdentityAwareProxyClient{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsBrandsIdentityAwareProxyClientsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes an Identity Aware Proxy (IAP) OAuth client. Useful for
// removing obsolete clients, managing the number of clients in a given
// project, and cleaning up after tests. Requires that the client is owned by
// IAP.
//
//   - name: Name of the Identity Aware Proxy client to be deleted. In the
//     following format:
//     projects/{project_number/id}/brands/{brand}/identityAwareProxyClients/{clie
//     nt_id}.
func (r *ProjectsBrandsIdentityAwareProxyClientsService) Delete(name string) *ProjectsBrandsIdentityAwareProxyClientsDeleteCall {
	c := &ProjectsBrandsIdentityAwareProxyClientsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsBrandsIdentityAwareProxyClientsDeleteCall) Fields(s ...googleapi.Field) *ProjectsBrandsIdentityAwareProxyClientsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsBrandsIdentityAwareProxyClientsDeleteCall) Context(ctx context.Context) *ProjectsBrandsIdentityAwareProxyClientsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsBrandsIdentityAwareProxyClientsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsBrandsIdentityAwareProxyClientsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.brands.identityAwareProxyClients.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsBrandsIdentityAwareProxyClientsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsBrandsIdentityAwareProxyClientsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves an Identity Aware Proxy (IAP) OAuth client. Requires that the
// client is owned by IAP.
//
//   - name: Name of the Identity Aware Proxy client to be fetched. In the
//     following format:
//     projects/{project_number/id}/brands/{brand}/identityAwareProxyClients/{clie
//     nt_id}.
func (r *ProjectsBrandsIdentityAwareProxyClientsService) Get(name string) *ProjectsBrandsIdentityAwareProxyClientsGetCall {
	c := &ProjectsBrandsIdentityAwareProxyClientsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsBrandsIdentityAwareProxyClientsGetCall) Fields(s ...googleapi.Field) *ProjectsBrandsIdentityAwareProxyClientsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsBrandsIdentityAwareProxyClientsGetCall) IfNoneMatch(entityTag string) *ProjectsBrandsIdentityAwareProxyClientsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsBrandsIdentityAwareProxyClientsGetCall) Context(ctx context.Context) *ProjectsBrandsIdentityAwareProxyClientsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsBrandsIdentityAwareProxyClientsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsBrandsIdentityAwareProxyClientsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.brands.identityAwareProxyClients.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *IdentityAwareProxyClient.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsBrandsIdentityAwareProxyClientsGetCall) Do(opts ...googleapi.CallOption) (*IdentityAwareProxyClient, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &IdentityAwareProxyClient{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsBrandsIdentityAwareProxyClientsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the existing clients for the brand.
//
//   - parent: Full brand path. In the following format:
//     projects/{project_number/id}/brands/{brand}.
func (r *ProjectsBrandsIdentityAwareProxyClientsService) List(parent string) *ProjectsBrandsIdentityAwareProxyClientsListCall {
	c := &ProjectsBrandsIdentityAwareProxyClientsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// clients to return. The service may return fewer than this value. If
// unspecified, at most 100 clients will be returned. The maximum value is
// 1000; values above 1000 will be coerced to 1000.
func (c *ProjectsBrandsIdentityAwareProxyClientsListCall) PageSize(pageSize int64) *ProjectsBrandsIdentityAwareProxyClientsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListIdentityAwareProxyClients` call. Provide this to
// retrieve the subsequent page. When paginating, all other parameters provided
// to `ListIdentityAwareProxyClients` must match the call that provided the
// page token.
func (c *ProjectsBrandsIdentityAwareProxyClientsListCall) PageToken(pageToken string) *ProjectsBrandsIdentityAwareProxyClientsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsBrandsIdentityAwareProxyClientsListCall) Fields(s ...googleapi.Field) *ProjectsBrandsIdentityAwareProxyClientsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsBrandsIdentityAwareProxyClientsListCall) IfNoneMatch(entityTag string) *ProjectsBrandsIdentityAwareProxyClientsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsBrandsIdentityAwareProxyClientsListCall) Context(ctx context.Context) *ProjectsBrandsIdentityAwareProxyClientsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsBrandsIdentityAwareProxyClientsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsBrandsIdentityAwareProxyClientsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/identityAwareProxyClients")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.brands.identityAwareProxyClients.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListIdentityAwareProxyClientsResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsBrandsIdentityAwareProxyClientsListCall) Do(opts ...googleapi.CallOption) (*ListIdentityAwareProxyClientsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListIdentityAwareProxyClientsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ProjectsBrandsIdentityAwareProxyClientsListCall) Pages(ctx context.Context, f func(*ListIdentityAwareProxyClientsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ProjectsBrandsIdentityAwareProxyClientsResetSecretCall struct {
	s                                          *Service
	name                                       string
	resetidentityawareproxyclientsecretrequest *ResetIdentityAwareProxyClientSecretRequest
	urlParams_                                 gensupport.URLParams
	ctx_                                       context.Context
	header_                                    http.Header
}

// ResetSecret: Resets an Identity Aware Proxy (IAP) OAuth client secret.
// Useful if the secret was compromised. Requires that the client is owned by
// IAP.
//
//   - name: Name of the Identity Aware Proxy client to that will have its secret
//     reset. In the following format:
//     projects/{project_number/id}/brands/{brand}/identityAwareProxyClients/{clie
//     nt_id}.
func (r *ProjectsBrandsIdentityAwareProxyClientsService) ResetSecret(name string, resetidentityawareproxyclientsecretrequest *ResetIdentityAwareProxyClientSecretRequest) *ProjectsBrandsIdentityAwareProxyClientsResetSecretCall {
	c := &ProjectsBrandsIdentityAwareProxyClientsResetSecretCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.resetidentityawareproxyclientsecretrequest = resetidentityawareproxyclientsecretrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsBrandsIdentityAwareProxyClientsResetSecretCall) Fields(s ...googleapi.Field) *ProjectsBrandsIdentityAwareProxyClientsResetSecretCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsBrandsIdentityAwareProxyClientsResetSecretCall) Context(ctx context.Context) *ProjectsBrandsIdentityAwareProxyClientsResetSecretCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsBrandsIdentityAwareProxyClientsResetSecretCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsBrandsIdentityAwareProxyClientsResetSecretCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.resetidentityawareproxyclientsecretrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:resetSecret")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.resetSecret", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.brands.identityAwareProxyClients.resetSecret" call.
// Any non-2xx status code is an error. Response headers are in either
// *IdentityAwareProxyClient.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsBrandsIdentityAwareProxyClientsResetSecretCall) Do(opts ...googleapi.CallOption) (*IdentityAwareProxyClient, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &IdentityAwareProxyClient{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.brands.identityAwareProxyClients.resetSecret", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsIapTunnelLocationsDestGroupsCreateCall struct {
	s               *Service
	parent          string
	tunneldestgroup *TunnelDestGroup
	urlParams_      gensupport.URLParams
	ctx_            context.Context
	header_         http.Header
}

// Create: Creates a new TunnelDestGroup.
//
//   - parent: Google Cloud Project ID and location. In the following format:
//     `projects/{project_number/id}/iap_tunnel/locations/{location}`.
func (r *ProjectsIapTunnelLocationsDestGroupsService) Create(parent string, tunneldestgroup *TunnelDestGroup) *ProjectsIapTunnelLocationsDestGroupsCreateCall {
	c := &ProjectsIapTunnelLocationsDestGroupsCreateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	c.tunneldestgroup = tunneldestgroup
	return c
}

// TunnelDestGroupId sets the optional parameter "tunnelDestGroupId": Required.
// The ID to use for the TunnelDestGroup, which becomes the final component of
// the resource name. This value must be 4-63 characters, and valid characters
// are `[a-z]-`.
func (c *ProjectsIapTunnelLocationsDestGroupsCreateCall) TunnelDestGroupId(tunnelDestGroupId string) *ProjectsIapTunnelLocationsDestGroupsCreateCall {
	c.urlParams_.Set("tunnelDestGroupId", tunnelDestGroupId)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsIapTunnelLocationsDestGroupsCreateCall) Fields(s ...googleapi.Field) *ProjectsIapTunnelLocationsDestGroupsCreateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsIapTunnelLocationsDestGroupsCreateCall) Context(ctx context.Context) *ProjectsIapTunnelLocationsDestGroupsCreateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsIapTunnelLocationsDestGroupsCreateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsIapTunnelLocationsDestGroupsCreateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.tunneldestgroup)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/destGroups")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.create", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.iap_tunnel.locations.destGroups.create" call.
// Any non-2xx status code is an error. Response headers are in either
// *TunnelDestGroup.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsIapTunnelLocationsDestGroupsCreateCall) Do(opts ...googleapi.CallOption) (*TunnelDestGroup, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TunnelDestGroup{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.create", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsIapTunnelLocationsDestGroupsDeleteCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// Delete: Deletes a TunnelDestGroup.
//
//   - name: Name of the TunnelDestGroup to delete. In the following format:
//     `projects/{project_number/id}/iap_tunnel/locations/{location}/destGroups/{d
//     est_group}`.
func (r *ProjectsIapTunnelLocationsDestGroupsService) Delete(name string) *ProjectsIapTunnelLocationsDestGroupsDeleteCall {
	c := &ProjectsIapTunnelLocationsDestGroupsDeleteCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsIapTunnelLocationsDestGroupsDeleteCall) Fields(s ...googleapi.Field) *ProjectsIapTunnelLocationsDestGroupsDeleteCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsIapTunnelLocationsDestGroupsDeleteCall) Context(ctx context.Context) *ProjectsIapTunnelLocationsDestGroupsDeleteCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsIapTunnelLocationsDestGroupsDeleteCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsIapTunnelLocationsDestGroupsDeleteCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.delete", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.iap_tunnel.locations.destGroups.delete" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsIapTunnelLocationsDestGroupsDeleteCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.delete", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsIapTunnelLocationsDestGroupsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Retrieves an existing TunnelDestGroup.
//
//   - name: Name of the TunnelDestGroup to be fetched. In the following format:
//     `projects/{project_number/id}/iap_tunnel/locations/{location}/destGroups/{d
//     est_group}`.
func (r *ProjectsIapTunnelLocationsDestGroupsService) Get(name string) *ProjectsIapTunnelLocationsDestGroupsGetCall {
	c := &ProjectsIapTunnelLocationsDestGroupsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsIapTunnelLocationsDestGroupsGetCall) Fields(s ...googleapi.Field) *ProjectsIapTunnelLocationsDestGroupsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsIapTunnelLocationsDestGroupsGetCall) IfNoneMatch(entityTag string) *ProjectsIapTunnelLocationsDestGroupsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsIapTunnelLocationsDestGroupsGetCall) Context(ctx context.Context) *ProjectsIapTunnelLocationsDestGroupsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsIapTunnelLocationsDestGroupsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsIapTunnelLocationsDestGroupsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.iap_tunnel.locations.destGroups.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *TunnelDestGroup.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsIapTunnelLocationsDestGroupsGetCall) Do(opts ...googleapi.CallOption) (*TunnelDestGroup, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TunnelDestGroup{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsIapTunnelLocationsDestGroupsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists the existing TunnelDestGroups. To group across all locations,
// use a `-` as the location ID. For example:
// `/v1/projects/123/iap_tunnel/locations/-/destGroups`
//
//   - parent: Google Cloud Project ID and location. In the following format:
//     `projects/{project_number/id}/iap_tunnel/locations/{location}`. A `-` can
//     be used for the location to group across all locations.
func (r *ProjectsIapTunnelLocationsDestGroupsService) List(parent string) *ProjectsIapTunnelLocationsDestGroupsListCall {
	c := &ProjectsIapTunnelLocationsDestGroupsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// PageSize sets the optional parameter "pageSize": The maximum number of
// groups to return. The service might return fewer than this value. If
// unspecified, at most 100 groups are returned. The maximum value is 1000;
// values above 1000 are coerced to 1000.
func (c *ProjectsIapTunnelLocationsDestGroupsListCall) PageSize(pageSize int64) *ProjectsIapTunnelLocationsDestGroupsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A page token, received
// from a previous `ListTunnelDestGroups` call. Provide this to retrieve the
// subsequent page. When paginating, all other parameters provided to
// `ListTunnelDestGroups` must match the call that provided the page token.
func (c *ProjectsIapTunnelLocationsDestGroupsListCall) PageToken(pageToken string) *ProjectsIapTunnelLocationsDestGroupsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsIapTunnelLocationsDestGroupsListCall) Fields(s ...googleapi.Field) *ProjectsIapTunnelLocationsDestGroupsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsIapTunnelLocationsDestGroupsListCall) IfNoneMatch(entityTag string) *ProjectsIapTunnelLocationsDestGroupsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsIapTunnelLocationsDestGroupsListCall) Context(ctx context.Context) *ProjectsIapTunnelLocationsDestGroupsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsIapTunnelLocationsDestGroupsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsIapTunnelLocationsDestGroupsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/destGroups")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.iap_tunnel.locations.destGroups.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListTunnelDestGroupsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsIapTunnelLocationsDestGroupsListCall) Do(opts ...googleapi.CallOption) (*ListTunnelDestGroupsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListTunnelDestGroupsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ProjectsIapTunnelLocationsDestGroupsListCall) Pages(ctx context.Context, f func(*ListTunnelDestGroupsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ProjectsIapTunnelLocationsDestGroupsPatchCall struct {
	s               *Service
	name            string
	tunneldestgroup *TunnelDestGroup
	urlParams_      gensupport.URLParams
	ctx_            context.Context
	header_         http.Header
}

// Patch: Updates a TunnelDestGroup.
//
//   - name: Identifier. Identifier for the TunnelDestGroup. Must be unique
//     within the project and contain only lower case letters (a-z) and dashes
//     (-).
func (r *ProjectsIapTunnelLocationsDestGroupsService) Patch(name string, tunneldestgroup *TunnelDestGroup) *ProjectsIapTunnelLocationsDestGroupsPatchCall {
	c := &ProjectsIapTunnelLocationsDestGroupsPatchCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.tunneldestgroup = tunneldestgroup
	return c
}

// UpdateMask sets the optional parameter "updateMask": A field mask that
// specifies which IAP settings to update. If omitted, then all of the settings
// are updated. See
// https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask
func (c *ProjectsIapTunnelLocationsDestGroupsPatchCall) UpdateMask(updateMask string) *ProjectsIapTunnelLocationsDestGroupsPatchCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsIapTunnelLocationsDestGroupsPatchCall) Fields(s ...googleapi.Field) *ProjectsIapTunnelLocationsDestGroupsPatchCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsIapTunnelLocationsDestGroupsPatchCall) Context(ctx context.Context) *ProjectsIapTunnelLocationsDestGroupsPatchCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsIapTunnelLocationsDestGroupsPatchCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsIapTunnelLocationsDestGroupsPatchCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.tunneldestgroup)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.patch", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.projects.iap_tunnel.locations.destGroups.patch" call.
// Any non-2xx status code is an error. Response headers are in either
// *TunnelDestGroup.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsIapTunnelLocationsDestGroupsPatchCall) Do(opts ...googleapi.CallOption) (*TunnelDestGroup, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TunnelDestGroup{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.projects.iap_tunnel.locations.destGroups.patch", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type V1GetIamPolicyCall struct {
	s                   *Service
	resource            string
	getiampolicyrequest *GetIamPolicyRequest
	urlParams_          gensupport.URLParams
	ctx_                context.Context
	header_             http.Header
}

// GetIamPolicy: Gets the access control policy for an Identity-Aware Proxy
// protected resource. More information about managing access via IAP can be
// found at:
// https://cloud.google.com/iap/docs/managing-access#managing_access_via_the_api
//
//   - resource: REQUIRED: The resource for which the policy is being requested.
//     See Resource names (https://cloud.google.com/apis/design/resource_names)
//     for the appropriate value for this field.
func (r *V1Service) GetIamPolicy(resource string, getiampolicyrequest *GetIamPolicyRequest) *V1GetIamPolicyCall {
	c := &V1GetIamPolicyCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resource = resource
	c.getiampolicyrequest = getiampolicyrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *V1GetIamPolicyCall) Fields(s ...googleapi.Field) *V1GetIamPolicyCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *V1GetIamPolicyCall) Context(ctx context.Context) *V1GetIamPolicyCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *V1GetIamPolicyCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *V1GetIamPolicyCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.getiampolicyrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+resource}:getIamPolicy")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resource": c.resource,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.getIamPolicy", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.getIamPolicy" call.
// Any non-2xx status code is an error. Response headers are in either
// *Policy.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *V1GetIamPolicyCall) Do(opts ...googleapi.CallOption) (*Policy, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Policy{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.getIamPolicy", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type V1GetIapSettingsCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetIapSettings: Gets the IAP settings on a particular IAP protected
// resource.
//
//   - name: The resource name for which to retrieve the settings. Authorization:
//     Requires the `getSettings` permission for the associated resource.
func (r *V1Service) GetIapSettings(name string) *V1GetIapSettingsCall {
	c := &V1GetIapSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *V1GetIapSettingsCall) Fields(s ...googleapi.Field) *V1GetIapSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *V1GetIapSettingsCall) IfNoneMatch(entityTag string) *V1GetIapSettingsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *V1GetIapSettingsCall) Context(ctx context.Context) *V1GetIapSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *V1GetIapSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *V1GetIapSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:iapSettings")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.getIapSettings", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.getIapSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *IapSettings.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *V1GetIapSettingsCall) Do(opts ...googleapi.CallOption) (*IapSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &IapSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.getIapSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type V1SetIamPolicyCall struct {
	s                   *Service
	resource            string
	setiampolicyrequest *SetIamPolicyRequest
	urlParams_          gensupport.URLParams
	ctx_                context.Context
	header_             http.Header
}

// SetIamPolicy: Sets the access control policy for an Identity-Aware Proxy
// protected resource. Replaces any existing policy. More information about
// managing access via IAP can be found at:
// https://cloud.google.com/iap/docs/managing-access#managing_access_via_the_api
//
//   - resource: REQUIRED: The resource for which the policy is being specified.
//     See Resource names (https://cloud.google.com/apis/design/resource_names)
//     for the appropriate value for this field.
func (r *V1Service) SetIamPolicy(resource string, setiampolicyrequest *SetIamPolicyRequest) *V1SetIamPolicyCall {
	c := &V1SetIamPolicyCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resource = resource
	c.setiampolicyrequest = setiampolicyrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *V1SetIamPolicyCall) Fields(s ...googleapi.Field) *V1SetIamPolicyCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *V1SetIamPolicyCall) Context(ctx context.Context) *V1SetIamPolicyCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *V1SetIamPolicyCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *V1SetIamPolicyCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.setiampolicyrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+resource}:setIamPolicy")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resource": c.resource,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.setIamPolicy", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.setIamPolicy" call.
// Any non-2xx status code is an error. Response headers are in either
// *Policy.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *V1SetIamPolicyCall) Do(opts ...googleapi.CallOption) (*Policy, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Policy{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.setIamPolicy", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type V1TestIamPermissionsCall struct {
	s                         *Service
	resource                  string
	testiampermissionsrequest *TestIamPermissionsRequest
	urlParams_                gensupport.URLParams
	ctx_                      context.Context
	header_                   http.Header
}

// TestIamPermissions: Returns permissions that a caller has on the
// Identity-Aware Proxy protected resource. More information about managing
// access via IAP can be found at:
// https://cloud.google.com/iap/docs/managing-access#managing_access_via_the_api
//
//   - resource: REQUIRED: The resource for which the policy detail is being
//     requested. See Resource names
//     (https://cloud.google.com/apis/design/resource_names) for the appropriate
//     value for this field.
func (r *V1Service) TestIamPermissions(resource string, testiampermissionsrequest *TestIamPermissionsRequest) *V1TestIamPermissionsCall {
	c := &V1TestIamPermissionsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.resource = resource
	c.testiampermissionsrequest = testiampermissionsrequest
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *V1TestIamPermissionsCall) Fields(s ...googleapi.Field) *V1TestIamPermissionsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *V1TestIamPermissionsCall) Context(ctx context.Context) *V1TestIamPermissionsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *V1TestIamPermissionsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *V1TestIamPermissionsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.testiampermissionsrequest)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+resource}:testIamPermissions")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"resource": c.resource,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.testIamPermissions", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.testIamPermissions" call.
// Any non-2xx status code is an error. Response headers are in either
// *TestIamPermissionsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *V1TestIamPermissionsCall) Do(opts ...googleapi.CallOption) (*TestIamPermissionsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &TestIamPermissionsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.testIamPermissions", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type V1UpdateIapSettingsCall struct {
	s           *Service
	name        string
	iapsettings *IapSettings
	urlParams_  gensupport.URLParams
	ctx_        context.Context
	header_     http.Header
}

// UpdateIapSettings: Updates the IAP settings on a particular IAP protected
// resource. It replaces all fields unless the `update_mask` is set.
//
// - name: The resource name of the IAP protected resource.
func (r *V1Service) UpdateIapSettings(name string, iapsettings *IapSettings) *V1UpdateIapSettingsCall {
	c := &V1UpdateIapSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.iapsettings = iapsettings
	return c
}

// UpdateMask sets the optional parameter "updateMask": The field mask
// specifying which IAP settings should be updated. If omitted, then all of the
// settings are updated. See
// https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask.
// Note: All IAP reauth settings must always be set together, using the field
// mask: `iapSettings.accessSettings.reauthSettings`.
func (c *V1UpdateIapSettingsCall) UpdateMask(updateMask string) *V1UpdateIapSettingsCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *V1UpdateIapSettingsCall) Fields(s ...googleapi.Field) *V1UpdateIapSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *V1UpdateIapSettingsCall) Context(ctx context.Context) *V1UpdateIapSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *V1UpdateIapSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *V1UpdateIapSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.iapsettings)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:iapSettings")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.updateIapSettings", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.updateIapSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *IapSettings.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *V1UpdateIapSettingsCall) Do(opts ...googleapi.CallOption) (*IapSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &IapSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.updateIapSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type V1ValidateAttributeExpressionCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// ValidateAttributeExpression: Validates that a given CEL expression conforms
// to IAP restrictions.
//
// - name: The resource name of the IAP protected resource.
func (r *V1Service) ValidateAttributeExpression(name string) *V1ValidateAttributeExpressionCall {
	c := &V1ValidateAttributeExpressionCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Expression sets the optional parameter "expression": Required. User input
// string expression. Should be of the form
// `attributes.saml_attributes.filter(attribute, attribute.name in
// ['{attribute_name}', '{attribute_name}'])`
func (c *V1ValidateAttributeExpressionCall) Expression(expression string) *V1ValidateAttributeExpressionCall {
	c.urlParams_.Set("expression", expression)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *V1ValidateAttributeExpressionCall) Fields(s ...googleapi.Field) *V1ValidateAttributeExpressionCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *V1ValidateAttributeExpressionCall) Context(ctx context.Context) *V1ValidateAttributeExpressionCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *V1ValidateAttributeExpressionCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *V1ValidateAttributeExpressionCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:validateAttributeExpression")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "iap.validateAttributeExpression", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "iap.validateAttributeExpression" call.
// Any non-2xx status code is an error. Response headers are in either
// *ValidateIapAttributeExpressionResponse.ServerResponse.Header or (if a
// response was returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *V1ValidateAttributeExpressionCall) Do(opts ...googleapi.CallOption) (*ValidateIapAttributeExpressionResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ValidateIapAttributeExpressionResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "iap.validateAttributeExpression", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}
