// Copyright 2022-2025 The sacloud/iaas-api-go Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"context"
	"log"
	"os"
	"path/filepath"

	"github.com/sacloud/iaas-api-go"
	"github.com/sacloud/iaas-api-go/helper/query"
	"github.com/sacloud/iaas-api-go/internal/tools"
	"github.com/sacloud/iaas-api-go/ostype"
	"github.com/sacloud/iaas-api-go/search"
	"github.com/sacloud/iaas-api-go/search/keys"
	"github.com/sacloud/iaas-api-go/types"
)

func init() {
	log.SetFlags(0)
	log.SetPrefix("gen-api-fake-data: ")
}

var fakeDataDefines = []struct {
	destination   string
	template      string
	parameterFunc func() interface{}
}{
	{
		destination:   "fake/zz_init_archive.go",
		template:      tmplArchive,
		parameterFunc: collectArchives,
	},
	{
		destination:   "fake/zz_init_cdrom.go",
		template:      tmplCDROM,
		parameterFunc: collectCDROMs,
	},
}

func main() {
	for _, generator := range fakeDataDefines {
		param := generator.parameterFunc()

		outputPath := filepath.Join(tools.ProjectRootPath(), generator.destination)

		tools.WriteFileWithTemplate(&tools.TemplateConfig{
			OutputPath: outputPath,
			Template:   generator.template,
			Parameter:  param,
		})

		log.Printf("generated: %s\n", outputPath)
	}
}

func collectArchives() interface{} {
	if os.Getenv("SAKURACLOUD_ACCESS_TOKEN") == "" ||
		os.Getenv("SAKURACLOUD_ACCESS_TOKEN_SECRET") == "" {
		log.Fatal("required: SAKURACLOUD_ACCESS_TOKEN and SAKURACLOUD_ACCESS_TOKEN_SECRET")
	}
	caller := iaas.NewClientFromEnv()
	tmplParam := map[string][]*iaas.Archive{}
	archiveOp := iaas.NewArchiveOp(caller)
	ctx := context.Background()

	for _, zone := range iaas.SakuraCloudZones {
		var archives []*iaas.Archive
		for _, ost := range ostype.ArchiveOSTypes {
			archive, err := query.FindArchiveByOSType(ctx, archiveOp, zone, ost)
			if err != nil {
				log.Fatal(err)
			}
			archives = append(archives, archive)
		}
		tmplParam[zone] = archives
	}
	return tmplParam
}

const tmplArchive = `// generated by 'github.com/sacloud/iaas-api-go/internal/tools/gen-api-fake-data'; DO NOT EDIT

package fake

import (
	"github.com/sacloud/iaas-api-go"
	"github.com/sacloud/iaas-api-go/types"
)

var initArchives = map[string][]*iaas.Archive{
{{ range $zone, $archives := . -}}
	"{{$zone}}": {
{{ range $archives -}}
		{
			ID:                   types.ID({{.ID}}),
			Name:                 "{{.Name}}",
			Description:          "fake",
			Tags:                 types.Tags{ {{range .Tags}}"{{.}}",{{ end }} },
			DisplayOrder:         {{.DisplayOrder}},
			Availability:         types.EAvailability("{{.Availability}}"),
			Scope:                types.EScope("{{.Scope}}"),
			SizeMB:               {{.SizeMB}},
			DiskPlanID:           types.ID({{.DiskPlanID}}),
			DiskPlanName:         "{{.DiskPlanName}}",
			DiskPlanStorageClass: "{{.DiskPlanStorageClass}}",
		},
{{ end -}}
	},
{{ end -}}
}
`

func collectCDROMs() interface{} {
	if os.Getenv("SAKURACLOUD_ACCESS_TOKEN") == "" ||
		os.Getenv("SAKURACLOUD_ACCESS_TOKEN_SECRET") == "" {
		log.Fatal("required: SAKURACLOUD_ACCESS_TOKEN and SAKURACLOUD_ACCESS_TOKEN_SECRET")
	}
	caller := iaas.NewClientFromEnv()

	tmplParam := map[string][]*iaas.CDROM{}
	cdromOp := iaas.NewCDROMOp(caller)
	ctx := context.Background()

	for _, zone := range iaas.SakuraCloudZones {
		var cdroms []*iaas.CDROM

		searched, err := cdromOp.Find(ctx, zone, &iaas.FindCondition{
			Filter: search.Filter{
				search.Key(keys.Scope): string(types.Scopes.Shared),
			},
		})
		if err != nil {
			log.Fatal(err)
		}

		cdroms = append(cdroms, searched.CDROMs...)
		tmplParam[zone] = cdroms
	}
	return tmplParam
}

const tmplCDROM = `// generated by 'github.com/sacloud/iaas-api-go/internal/tools/gen-api-fake-data'; DO NOT EDIT

package fake

import (
	"github.com/sacloud/iaas-api-go"
	"github.com/sacloud/iaas-api-go/types"
)

var initCDROM = map[string][]*iaas.CDROM{
{{ range $zone, $data := . -}}
	"{{$zone}}": {
{{ range $data -}}
		{
			ID:                   types.ID({{.ID}}),
			Name:                 "{{.Name}}",
			Description:          "fake",
			Tags:                 types.Tags{ {{range .Tags}}"{{.}}",{{ end }} },
			DisplayOrder:         {{.DisplayOrder}},
			Availability:         types.EAvailability("{{.Availability}}"),
			Scope:                types.EScope("{{.Scope}}"),
			SizeMB:               {{.SizeMB}},
		},
{{ end -}}
	},
{{ end -}}
}
`
