package appsec

import (
	"context"
	"encoding/json"
	"errors"
	"net/http"
	"net/http/httptest"
	"testing"

	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

// Test MalwareContentTypes
func TestAppSec_GetMalwareContentTypes(t *testing.T) {

	result := GetMalwareContentTypesResponse{}

	respData := compactJSON(loadFixtureBytes("testdata/TestMalwareContentTypes/MalwareContentTypes.json"))
	err := json.Unmarshal([]byte(respData), &result)
	require.NoError(t, err)

	tests := map[string]struct {
		params           GetMalwareContentTypesRequest
		responseStatus   int
		responseBody     string
		expectedPath     string
		expectedResponse *GetMalwareContentTypesResponse
		withError        error
	}{
		"200 OK": {
			params: GetMalwareContentTypesRequest{
				ConfigID: 43253,
				Version:  15,
			},
			responseStatus:   http.StatusOK,
			responseBody:     respData,
			expectedPath:     "/appsec/v1/configs/43253/versions/15/malware-policies/content-types",
			expectedResponse: &result,
		},
		"500 internal server error": {
			params: GetMalwareContentTypesRequest{
				ConfigID: 43253,
				Version:  15,
			},
			responseStatus: http.StatusInternalServerError,
			responseBody: (`
{
    "type": "internal_error",
    "title": "Internal Server Error",
    "detail": "Error fetching MalwareContentTypes"
}`),
			expectedPath: "/appsec/v1/configs/43253/versions/15/malware-policies/content-types",
			withError: &Error{
				Type:       "internal_error",
				Title:      "Internal Server Error",
				Detail:     "Error fetching MalwareContentTypes",
				StatusCode: http.StatusInternalServerError,
			},
		},
	}

	for name, test := range tests {
		t.Run(name, func(t *testing.T) {
			mockServer := httptest.NewTLSServer(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
				assert.Equal(t, test.expectedPath, r.URL.String())
				assert.Equal(t, http.MethodGet, r.Method)
				w.WriteHeader(test.responseStatus)
				_, err := w.Write([]byte(test.responseBody))
				assert.NoError(t, err)
			}))
			client := mockAPIClient(t, mockServer)
			result, err := client.GetMalwareContentTypes(context.Background(), test.params)
			if test.withError != nil {
				assert.True(t, errors.Is(err, test.withError), "want: %s; got: %s", test.withError, err)
				return
			}
			require.NoError(t, err)
			assert.Equal(t, test.expectedResponse, result)
		})
	}
}
