package services

import (
	"context"
	"fmt"
	"os"

	"github.com/lxc/incus/v6/shared/subprocess"

	"github.com/lxc/incus-os/incus-osd/api"
	"github.com/lxc/incus-os/incus-osd/internal/state"
	"github.com/lxc/incus-os/incus-osd/internal/systemd"
)

var ovnSystemdTLS = `# Systemd unit generated by Incus OS
[Unit]
Description=Open Virtual Network host control daemon

[Service]
ExecStart=/usr/bin/ovn-controller --private-key=/run/ovn/client.key --certificate=/run/ovn/client.crt --ca-cert=/run/ovn/ca.crt
Restart=on-failure
`

var ovnSystemd = `# Systemd unit generated by Incus OS
[Unit]
Description=Open Virtual Network host control daemon

[Service]
ExecStart=/usr/bin/ovn-controller
Restart=on-failure
`

// OVN represents the system OVS/OVN service.
type OVN struct {
	common

	state *state.State
}

// Get returns the current service state.
func (n *OVN) Get(_ context.Context) (any, error) {
	return n.state.Services.OVN, nil
}

// Update updates the service configuration.
func (n *OVN) Update(ctx context.Context, req any) error {
	newState, ok := req.(*api.ServiceOVN)
	if !ok {
		return fmt.Errorf("request type \"%T\" isn't expected ServiceOVN", req)
	}

	// Save the state on return.
	defer n.state.Save(ctx)

	// Disable the service if requested.
	if n.state.Services.OVN.Config.Enabled && !newState.Config.Enabled {
		err := n.Stop(ctx)
		if err != nil {
			return err
		}
	}

	// Update the configuration.
	n.state.Services.OVN.Config = newState.Config

	// Enable the service if requested.
	if !n.state.Services.OVN.Config.Enabled && newState.Config.Enabled {
		err := n.Start(ctx)
		if err != nil {
			return err
		}
	}

	// Configure the service.
	err := n.configure(ctx)
	if err != nil {
		return err
	}

	return nil
}

// Stop stops the service.
func (n *OVN) Stop(ctx context.Context) error {
	if !n.state.Services.OVN.Config.Enabled {
		return nil
	}

	// Stop OVS and OVN.
	err := systemd.StopUnit(ctx, "ovn-controller.service", "ovs-vswitchd.service", "ovsdb-server.service")
	if err != nil {
		return err
	}

	return nil
}

// Start starts the service.
func (n *OVN) Start(ctx context.Context) error {
	if !n.state.Services.OVN.Config.Enabled {
		return nil
	}

	// Start OVS.
	err := systemd.StartUnit(ctx, "ovs-vswitchd.service")
	if err != nil {
		return err
	}

	// Configure OVS and bring up OVN.
	return n.configure(ctx)
}

// ShouldStart returns true if the service should be started on boot.
func (n *OVN) ShouldStart() bool {
	return n.state.Services.OVN.Config.Enabled
}

// Struct returns the API struct for the OVN service.
func (*OVN) Struct() any {
	return &api.ServiceOVN{}
}

// configure takes care of configuring the running OVS and (re)spawning the OVN controller.
func (n *OVN) configure(ctx context.Context) error {
	// Apply the OVS configuration.
	args := []string{"set", "open_vswitch", "."}

	args = append(args, "external_ids:hostname="+n.state.Hostname())
	args = append(args, "external_ids:ovn-remote="+n.state.Services.OVN.Config.Database)
	args = append(args, "external_ids:ovn-encap-type="+n.state.Services.OVN.Config.TunnelProtocol)
	args = append(args, "external_ids:ovn-encap-ip="+n.state.Services.OVN.Config.TunnelAddress)
	args = append(args, fmt.Sprintf("external_ids:ovn-is-interconn=%v", n.state.Services.OVN.Config.ICChassis))

	_, err := subprocess.RunCommand("ovs-vsctl", args...)
	if err != nil {
		return err
	}

	// Write the OVN certificates (if provided).
	err = os.MkdirAll("/run/ovn", 0o700)
	if err != nil {
		return err
	}

	if n.state.Services.OVN.Config.TLSClientCertificate != "" {
		err = os.WriteFile("/run/ovn/client.crt", []byte(n.state.Services.OVN.Config.TLSClientCertificate), 0o600)
		if err != nil {
			return err
		}
	}

	if n.state.Services.OVN.Config.TLSClientKey != "" {
		err = os.WriteFile("/run/ovn/client.key", []byte(n.state.Services.OVN.Config.TLSClientKey), 0o600)
		if err != nil {
			return err
		}
	}

	if n.state.Services.OVN.Config.TLSCACertificate != "" {
		err = os.WriteFile("/run/ovn/ca.crt", []byte(n.state.Services.OVN.Config.TLSCACertificate), 0o600)
		if err != nil {
			return err
		}
	}

	// Generate the systemd unit.
	if n.state.Services.OVN.Config.TLSClientCertificate != "" {
		err = os.WriteFile("/run/systemd/system/ovn-controller.service", []byte(ovnSystemdTLS), 0o600)
		if err != nil {
			return err
		}
	} else {
		err = os.WriteFile("/run/systemd/system/ovn-controller.service", []byte(ovnSystemd), 0o600)
		if err != nil {
			return err
		}
	}

	err = systemd.ReloadDaemon(ctx)
	if err != nil {
		return err
	}

	// (Re)start the OVN controller.
	err = systemd.RestartUnit(ctx, "ovn-controller.service")
	if err != nil {
		return err
	}

	return nil
}
