package bmp

// This code uses BMP images generated by BMP Suite (https://entropymine.com/jason/bmpsuite/).

import (
	"bytes"
	"encoding/binary"
	"fmt"
	"image"
	"image/color"
	"image/png"
	"io/ioutil"
	"path/filepath"
	"strings"
	"testing"
)

func TestDecode(t *testing.T) {
	files, err := filepath.Glob("testdata/*.png")
	if err != nil {
		panic("failed to list test files: " + err.Error())
	}
	for _, file := range files {
		t.Run(file, func(t *testing.T) {
			in, err := ioutil.ReadFile(file)
			if err != nil {
				panic("failed to read " + file + ": " + err.Error())
			}
			img, err := png.Decode(bytes.NewReader(in))
			if err != nil {
				t.Fatalf("png.Decode() = _, %v; want nil", err)
			}
			bmpFile := strings.Replace(file, ".png", ".bmp", 1)
			bmpIn, err := ioutil.ReadFile(bmpFile)
			if err != nil {
				panic("failed to read " + bmpFile + ": " + err.Error())
			}
			bmpImg, err := Decode(bytes.NewReader(bmpIn))
			if err != nil {
				t.Fatalf("Decode() = _, %v; want nil", err)
			}
			switch file {
			case "testdata/rgb16.png", "testdata/rgb16bfdef.png":
				img = rgb5x5Image{
					Image:  img,
					rgb565: false,
				}
			case "testdata/rgb16-565.png":
				img = rgb5x5Image{
					Image:  img,
					rgb565: true,
				}
			}
			compare(t, img, bmpImg)
		})
	}
}

type rgb5x5Image struct {
	image.Image
	rgb565 bool
}

func (p rgb5x5Image) At(x, y int) color.Color {
	c := p.Image.At(x, y).(color.RGBA)
	c.R &= 0xF8
	if p.rgb565 {
		c.G &= 0xFC
	} else {
		c.G &= 0xF8
	}
	c.B &= 0xF8
	return c
}

func compare(t *testing.T, expected, actual image.Image) {
	if !expected.Bounds().Eq(actual.Bounds()) {
		t.Fatalf("Bounds() = %s; want %s", actual.Bounds(), expected.Bounds())
	}
	errors := 0
	for y := expected.Bounds().Min.Y; y < expected.Bounds().Max.Y; y++ {
		for x := expected.Bounds().Min.X; x < expected.Bounds().Max.X; x++ {
			expectedR, expectedG, expectedB, expectedA := expected.At(x, y).RGBA()
			actualR, actualG, actualB, actualA := actual.At(x, y).RGBA()
			if expectedR != actualR || expectedG != actualG || expectedB != actualB || expectedA != actualA {
				t.Errorf("At(%d, %d) = %v; want %v", x, y, actual.At(x, y), expected.At(x, y))
				if errors >= 50 {
					t.Fatalf("Too many At() errors")
				} else {
					errors++
				}
			}
		}
	}
}

func TestDecodeShouldFail(t *testing.T) {
	b := make([]byte, 1024)
	expect := func(t *testing.T, msg string) {
		_, err := Decode(bytes.NewReader(b))
		if err == nil || err.Error() != msg {
			t.Fatalf("Decoder() = _, %v; want %s", err, msg)
		}
	}
	expect(t, "bmp: invalid format: not a BMP file")
	b[0], b[1] = 'B', 'M'
	expect(t, "bmp: unsupported feature: DIB header version")
	binary.LittleEndian.PutUint32(b[14:], 40)
	expect(t, "bmp: unsupported feature: planes 0")
	binary.LittleEndian.PutUint32(b[18:], 1<<31)
	expect(t, "bmp: unsupported feature: non-positive dimension")
	binary.LittleEndian.PutUint32(b[18:], 1024)
	binary.LittleEndian.PutUint32(b[22:], 1024)
	binary.LittleEndian.PutUint16(b[26:], 1)
	expect(t, "bmp: unsupported feature: bit depth 0")
	binary.LittleEndian.PutUint16(b[30:], 1)
	expect(t, "bmp: unsupported feature: compression method")
	binary.LittleEndian.PutUint16(b[30:], 3)
	expect(t, "bmp: unsupported feature: compression method")
	binary.LittleEndian.PutUint16(b[30:], 0)
	for _, bpp := range []uint16{1, 2, 4, 8, 16, 24, 32} {
		t.Run(fmt.Sprintf("bpp=%d", bpp), func(t *testing.T) {
			binary.LittleEndian.PutUint32(b[10:], 0)
			binary.LittleEndian.PutUint16(b[28:], bpp)
			expect(t, "bmp: unsupported feature: bitmap offset")
			offset := uint32(fileHeaderLen + infoHeaderLen)
			if bpp < 16 {
				offset += 4 * 1 << bpp
			}
			binary.LittleEndian.PutUint32(b[10:], offset)
			expect(t, "unexpected EOF")
		})
	}
}
