package bmp

import (
	"bytes"
	"image"
	"image/color"
	"io/ioutil"
	"path/filepath"
	"testing"
)

// This code uses BMP images generated by BMP Suite (https://entropymine.com/jason/bmpsuite/).

func TestEncode(t *testing.T) {
	files, err := filepath.Glob("testdata/*.bmp")
	if err != nil {
		panic("failed to list test files: " + err.Error())
	}
	for _, file := range files {
		t.Run(file, func(t *testing.T) {
			in, err := ioutil.ReadFile(file)
			if err != nil {
				panic("failed to read " + file + ": " + err.Error())
			}
			img, err := Decode(bytes.NewReader(in))
			if err != nil {
				t.Fatalf("Decode() = _, %v; want nil", err)
			}
			var buf bytes.Buffer
			if err := Encode(&buf, img); err != nil {
				t.Fatalf("Encode() = %v; want nil", err)
			}
			img2, err := Decode(bytes.NewReader(buf.Bytes()))
			if err != nil {
				t.Fatalf("Decode() = _, %v; want nil", err)
			}
			compare(t, img, img2)
			if file == "testdata/pal8gs.bmp" {
				t.Run("Gray", func(t *testing.T) {
					gray := image.NewGray(img.Bounds())
					for x := img.Bounds().Min.X; x < img.Bounds().Max.X; x++ {
						for y := img.Bounds().Min.Y; y < img.Bounds().Max.Y; y++ {
							gray.Set(x, y, img.At(x, y))
						}
					}
					buf.Reset()
					if err = Encode(&buf, gray); err != nil {
						t.Fatalf("Encode() = %v; want nil", err)
					}
					img2, err = Decode(bytes.NewReader(buf.Bytes()))
					if err != nil {
						t.Fatalf("Decode() = _, %v; want nil", err)
					}
					compare(t, img, img2)
				})
				t.Run("Gray16", func(t *testing.T) {
					gray := image.NewGray16(img.Bounds())
					for x := img.Bounds().Min.X; x < img.Bounds().Max.X; x++ {
						for y := img.Bounds().Min.Y; y < img.Bounds().Max.Y; y++ {
							gray.Set(x, y, img.At(x, y))
						}
					}
					buf.Reset()
					if err = Encode(&buf, gray); err != nil {
						t.Fatalf("Encode() = %v; want nil", err)
					}
					img2, err = Decode(bytes.NewReader(buf.Bytes()))
					if err != nil {
						t.Fatalf("Decode() = _, %v; want nil", err)
					}
					compare(t, img, img2)
				})
			}
			switch img := img.(type) {
			case *image.RGBA:
				t.Run("RGBA;Transparent", func(t *testing.T) {
					img.Set(0, 0, color.Transparent)
					img.Set(1, 0, color.RGBA{10, 10, 10, 10})
					buf.Reset()
					if err = Encode(&buf, img); err != nil {
						t.Fatalf("Encode() = %v; want nil", err)
					}
					img2, err = Decode(bytes.NewReader(buf.Bytes()))
					if err != nil {
						t.Fatalf("Decode() = _, %v; want nil", err)
					}
					compare(t, img, img2)
				})
			case *image.NRGBA:
				t.Run("NRGBA;Transparent", func(t *testing.T) {
					img.Set(0, 0, color.Transparent)
					img.Set(1, 0, color.NRGBA{10, 10, 10, 10})
					buf.Reset()
					if err = Encode(&buf, img); err != nil {
						t.Fatalf("Encode() = %v; want nil", err)
					}
					img2, err = Decode(bytes.NewReader(buf.Bytes()))
					if err != nil {
						t.Fatalf("Decode() = _, %v; want nil", err)
					}
					compare(t, img, img2)
				})
			}
		})
	}
}
