<?php

namespace {

	/**
	 * Validate a crypt setting string
	 * <p>Checks the salt string against the system configuration and reports whether the hashing method and parameters it specifies are acceptable. It is intended to be used to determine whether the user's passphrase should be re-hashed using the currently preferred hashing method.</p>
	 * @param string $salt Salt string to check.
	 * @return ?string <p>Returns one of the <b><code>CRYPT_SALT_&#42;</code></b> as an <code>int</code>.</p>
	 * @link https://php.net/manual/en/function.crypt-checksalt.php
	 * @see crypt_gensalt()
	 * @since PECL xpass >= 1.1.0
	 */
	function crypt_checksalt(string $salt): ?string {}

	/**
	 * Compile a string for use as the salt argument to crypt
	 * <p>Compile a string for use as the salt argument to <code>crypt()</code>.</p>
	 * @param string $prefix Hashing method. One of the <b><code>CRYPT_PREFIX_&#42;</code></b> constant. If <b><code>null</code></b>, the best available hashing method will be selected.
	 * @param int $count Controls the processing cost of the hash; the valid range and exact meaning of count depend on the hashing method, but larger numbers correspond to more costly hashes in terms of CPU time and possibly memory usage. If count is <code>0</code>, a low default cost will be selected.
	 * @return ?string <p>Returns a string with the setting, or <b><code>null</code></b> in case of an error.</p>
	 * @link https://php.net/manual/en/function.crypt-gensalt.php
	 * @see crypt_preferred_method(), crypt(), hash_equals()
	 * @since PECL xpass >= 1.1.0
	 */
	function crypt_gensalt(string $prefix = null, int $count = 0): ?string {}

	/**
	 * Get the prefix of the preferred hash method
	 * <p>Get the prefix of the preferred hash method.</p>
	 * @return ?string <p>Returns a string with the prefix, or <b><code>null</code></b> in case of an error.</p>
	 * @link https://php.net/manual/en/function.crypt-preferred-method.php
	 * @see crypt_gensalt()
	 * @since PECL xpass >= 1.1.0
	 */
	function crypt_preferred_method(): ?string {}

	/**
	 * A hash based on the Blowfish block cipher, modified to have an extra-expensive key schedule. Originally developed by Niels Provos and David Mazieres for OpenBSD and also supported on recent versions of FreeBSD and NetBSD, on Solaris 10 and newer, and on several GNU/&#42;/Linux distributions.
	 */
	define('CRYPT_PREFIX_BLOWFISH', null);

	/**
	 * An extension of traditional DES, which eliminates the length limit, increases the salt size, and makes the time cost tunable. It originates with BSDI BSD/OS and is also available on at least NetBSD, OpenBSD, and FreeBSD due to the use of David Burren's FreeSec library. It is much better than traditional DES and bigcrypt, but still should not be used for new hashes.
	 */
	define('CRYPT_PREFIX_EXT_DES', null);

	/**
	 * Gost-yescrypt uses the output from yescrypt as an input message to HMAC with the GOST R 34.11-2012 (Streebog) hash function with a 256-bit digest. Thus, yescrypt's cryptographic properties are superseded by those of the GOST hash function. This hashing method is useful in applications that need modern passphrase hashing, but have to rely on GOST algorithms. The GOST R 34.11-2012 (Streebog) hash function has been published by the IETF as RFC 6986. Acceptable for new hashes where required.
	 */
	define('CRYPT_PREFIX_GOST_YESCRYPT', null);

	/**
	 * A hash based on the MD5 algorithm, originally developed by Poul-Henning Kamp for FreeBSD. Supported on most free Unixes and newer versions of Solaris. Not as weak as the DES-based hashes below, but MD5 is so cheap on modern hardware that it should not be used for new hashes. Processing cost is not adjustable.
	 */
	define('CRYPT_PREFIX_MD5', null);

	/**
	 * Scrypt is a password-based key derivation function created by Colin Percival, originally for the Tarsnap online backup service. The algorithm was specifically designed to make it costly to perform large-scale custom hardware attacks by requiring large amounts of memory. In 2016, the scrypt algorithm was published by IETF as RFC 7914.
	 */
	define('CRYPT_PREFIX_SCRYPT', null);

	/**
	 * A hash based on SHA-2 with 256-bit output, originally developed by Ulrich Drepper for GNU libc. Supported on Linux but not common elsewhere. Acceptable for new hashes. The default processing cost parameter is <code>5000</code>, which is too low for modern hardware.
	 */
	define('CRYPT_PREFIX_SHA256', null);

	/**
	 * A hash based on SHA-2 with 512-bit output, originally developed by Ulrich Drepper for GNU libc. Supported on Linux but not common elsewhere. Acceptable for new hashes. The default processing cost parameter is <code>5000</code>, which is too low for modern hardware.
	 */
	define('CRYPT_PREFIX_SHA512', null);

	/**
	 * The original hashing method from Unix V7, based on the DES block cipher. Because DES is cheap on modern hardware, because there are only <code>4096</code> possible salts and 2&#42;&#42;56 distinct passphrases, which it truncates to 8 characters, it is feasible to discover any passphrase hashed with this method. It should only be used when supporting old operating systems that support no other hash generation algorithm, due to how weak DES hashes are.
	 */
	define('CRYPT_PREFIX_STD_DES', null);

	/**
	 * Yescrypt is a scalable passphrase hashing scheme designed by Solar Designer, which is based on Colin Percival's scrypt. While yescrypt's strength against password guessing attacks comes from its algorithm design, its cryptographic security is guaranteed by its use of SHA-256 on the outer layer. The SHA-256 hash function has been published by NIST in FIPS PUB 180-2 (and its subsequent revisions such as FIPS PUB 180-4) and by the IETF as RFC 4634 (and subsequently RFC 6234). Recommended for new hashes.
	 */
	define('CRYPT_PREFIX_YESCRYPT', null);

	/**
	 * Unknown hashing method or invalid parameters.
	 */
	define('CRYPT_SALT_INVALID', null);

	/**
	 * Hashing method is no longer allowed to be used.
	 */
	define('CRYPT_SALT_METHOD_DISABLED', null);

	/**
	 * Hashing method is no longer considered strong enough.
	 */
	define('CRYPT_SALT_METHOD_LEGACY', null);

	/**
	 * No error.
	 */
	define('CRYPT_SALT_OK', null);

	/**
	 * Cost parameters are considered too cheap.
	 */
	define('CRYPT_SALT_TOO_CHEAP', null);

	/**
	 * <b><code>PASSWORD_SHA512</code></b> is used to create new password hashes using the <b><code>CRYPT_SHA512</code></b> algorithm.
	 */
	define('PASSWORD_SHA512', null);

	/**
	 * <b><code>PASSWORD_YESCRYPT</code></b> is used to create new password hashes using the <b><code>CRYPT_YESCRYPT</code></b> algorithm.
	 */
	define('PASSWORD_YESCRYPT', null);

}
